require import FSet Int IntExtra Real SmtMap Distr.

require import Timestamp.
require import Tags.

theory BLT_Scheme_Theory.

type pkey, skey, tag, message, data, hash_output.

op keyGen : (pkey * skey) distr.
op tagGen : skey -> int -> tag.
op tagVer : pkey -> int -> tag -> bool.

op bind   : message * tag -> data.

op tdistr : int distr.
op H : tag -> hash_output.


clone export Timestamping as R with type data   <- data,
                                    op   tdistr <- tdistr.


clone export TagSystem as T with type pkey          <- pkey,
                                 type skey          <- skey,
                                 type tag           <- tag,
                                 type hash_output   <- hash_output,
                                 op   keyGen        <- keyGen,
                                 op   tagGen        <- tagGen,
                                 op   tagVer        <- tagVer, 
                                 op   H             <- H.


(* BLT scheme implementation parameterized by timestamping and tagging oracles  *)
module BLTScheme(TsO : TS, TagO : TagOracleT) = {

  proc sign(m : message) : tag * int = {
    var t, tg, d;
 
    t  = TsO.clock();
    tg = TagO.genTag(t+1);
    d  = bind (m, oget tg);
    TsO.put(d);

    return (oget tg, t+1); 
  }

  proc verify(m : message, tg : tag, t : int) : bool = {
    var valid, timestamped;

    valid       = TagO.verTag(tg, t);
    timestamped = TsO.check(t, bind (m, tg));

    return valid /\ timestamped;  
  }
}.


module type BLTOracleT = {
  proc * init(pk : pkey, sk : skey) : unit
  proc sign(m : message) : (tag * int) option
  proc verify(m : message, tg : tag, t : int) : bool
  proc fresh(m : message) : bool
}.


(* BLT adversaries and the existential unforgeability (EUF-CMA) game *)
module type AdvBLT (T : TS, O : BLTOracleT) = {
  proc forge(pk : pkey) : message * tag * int {O.sign T.put T.check}
}.


module type AdvBLTRO (T : TS, O : BLTOracleT) = {
  proc forge(pk : pkey) : message * tag * int {O.sign T.check}
}.


module GameBLT(BLTO : BLTOracleT, A : AdvBLT) = {
  module A = A(Ts, BLTO)

  var tg : tag
  var m : message
  var t : int

  proc main() : bool = {
    var pk, sk, forged, fresh;

    (pk, sk) =$ keyGen;
    BLTO.init(pk, sk);
    
    (m, tg, t) = A.forge(pk);

    forged  = BLTO.verify(m, tg, t);
    fresh   = BLTO.fresh(m);

    return forged /\ fresh;
  }
}.


(* standard implementation of a BLT oracle *)
module BLTOracle : BLTOracleT = {
  module BLT = BLTScheme(Ts, TagOracle)

  var qs   : message option
  var qt   : int
  var used : bool

  proc init(pk : pkey, sk : skey) : unit = {

    TagOracle.init(pk, sk);
    Ts.init();

    qt   = 0;
    qs   = None;
    used = false;
  }

  proc sign(m : message) : (tag * int) option = {
    var r, q;

    if(!used){
      qs = Some m;
      (q, qt) = BLT.sign(m);
      r = Some (q, qt);
    }else{
      r = None;
    }
    used = true;

    return r;
  }

  proc verify(m : message, tg : tag, t : int):bool = {
      var b : bool;
      b = BLT.verify(m, tg, t);
      return b;    
  }

  proc fresh(m : message) : bool = {
    return qs <> Some m;
  }

}.

(* Dummy oracle which never signs a single message *)
module BLTDummy : BLTOracleT = {

  module BLT = BLTScheme(Ts, TagOracle)

  proc init(pk : pkey, sk : skey) : unit = {
    TagOracle.init(pk, sk);
    Ts.init();
  }

  proc sign(m : message) : (tag * int) option = {
     return None;
  }

  proc verify(m : message, tg : tag, t : int) : bool = {
      var b : bool;
      b = BLT.verify(m, tg, t);
      return b;    
  }

  proc fresh(m : message) : bool = {
    return true;
  }
}.



(* Correctness *)
module BLTCorrect = {
   module BLT = BLTScheme(Ts, TagOracle)

   proc main(m : message) = {
      var pk, sk, t, tg, r;

      (pk, sk) =$ keyGen;
      Ts.init();
      TagOracle.init(pk, sk);

      (tg, t) = BLT.sign(m);
      r = BLT.verify(m, tg, t);
     
      return r;  
  }

}.

(* Correctness of one-time BLT scheme *)

(* Correcetness of BLT when combined with unbounded tag system *)
lemma bltCorrectU : 
  is_lossless keyGen => is_lossless tdistr
   => (forall pk sk t,  (pk, sk) \in keyGen => tagVer pk t (tagGen sk t) = true) =>
  phoare [ BLTCorrect.main : true ==> res ] = 1%r.
proof. move => lk lt vt. 
proc. inline*. wp. rnd predT. wp. rnd predT. skip. progress. smt. smt.  smt.
smt. 
qed.


(* Correctness of BLT when combined with bounded tag system *)
op kpe : int.
axiom kpetdist t : t \in tdistr => t < kpe.

lemma bltCorrectB : 
  is_lossless keyGen => is_lossless tdistr
   => (forall pk sk t,  (pk, sk) \in keyGen => (tagVer pk t (tagGen sk t) = true <=> t <= kpe)) =>
  phoare [ BLTCorrect.main : true ==> res ] = 1%r.
proof. move => lk lt vt. 
proc. inline*. wp. rnd predT. wp. rnd predT. skip. progress. smt. smt.  smt.
smt. 
qed.


(* Properties of a Dummy Oracle *)
axiom tdistr_pos t : t \in tdistr => 0 < t. 

module N(A:AdvBLT, O:TagOracleT) = {
  module A = A(Ts, BLTDummy)

  proc forge(pk:pkey) = {
   var tg, m, t;
   Ts.init();
   (m, tg, t) = A.forge(pk);

   return (tg, t);
  }
}.

section.

declare module A : AdvBLT{Ts, TagOracle}.

local lemma d2f : 
  equiv [ GameBLT(BLTDummy, A).main ~ GameFR(TagOracle, N(A)).main
  : ={glob A} ==>  res{1} => res{2} ].

proof. proc.
inline*. wp.
call (_: ={glob TagOracle, glob Ts} 
  /\ (forall x, x \in Ts.r{1} => 0 < x) 
  /\ 0 <= Ts.i{1} 
  /\ Ts.i{1} <= Ts.t{1} ).

proc. skip. smt.
proc. inline*. wp. skip. progress;smt.
proc. inline*. wp. skip. progress. 
wp. swap {2} 6 -5. wp. rnd.
wp. rnd. wp.  skip. progress;smt.
qed.


lemma d2f_pr &m : 
  Pr[ GameBLT(BLTDummy, A).main() @ &m : res ] <=
    Pr[ GameFR(TagOracle, N(A)).main() @ &m : res ].
proof. byequiv.  conseq d2f;smt. auto. auto. qed.

end section.



end BLT_Scheme_Theory.



