require import Int Bool Distr.

abstract theory TagSystem.


(* Parameter types and operators *)
type pkey, skey, tag.

op keyGen : (pkey * skey) distr.
op tagGen : skey -> int -> tag.
op tagVer : pkey -> int -> tag -> bool.


(* Interface of One-Time Tag Oracle *)
module type TagOracleT = {
  proc * init(pk : pkey, sk : skey) : unit
  proc genTag(t : int)              : tag option
  proc verTag(tg : tag, t : int)    : bool
  proc usedTime()                   : int
}.



(* Standard implementation of One-Time Tag Oracle *)
module TagOracle : TagOracleT = {

  var usedFlag : bool
  var usedTime  : int
  var pk       : pkey
  var sk       : skey

  proc init(pk : pkey, sk : skey) : unit = {

    TagOracle.pk = pk;
    TagOracle.sk = sk;
    usedFlag     = false;
    usedTime      = 0;

  }

  proc genTag(t : int) : tag option = {

    var r = None;
    if(!usedFlag){
      usedTime = t;
      r = Some (tagGen sk t);
    }
    usedFlag = true;

    return r;
  }

  proc verTag(tg : tag, t : int) : bool = {
    return tagVer pk t tg;
  }

  proc usedTime() : int = {
    return usedTime;
  } 
}.


(* Forward-Resistance *)
module type AdvFR(TagO : TagOracleT) = {
  proc forge(pk : pkey) : tag * int  {TagO.genTag}
}.

module GameFR(TagO : TagOracleT, A : AdvFR) = {
  module A = A(TagO)

  proc main() : bool = {
    var pk, sk, tg, t, t', forged;

    (pk, sk) <$ keyGen;
    
    TagO.init(pk, sk);
    (tg, t) = A.forge(pk);
    
    forged = TagO.verTag(tg, t);
    t'     = TagO.usedTime();    

    return forged /\ t' < t;
  }
}.


(* Hash-and-Tag Unpredictability (adversary and game) *)
type hash_output.
op H : tag -> hash_output.

module type AdvTHU = {
  proc forge(pk : pkey) : hash_output * int
}.

module GameTHU(A : AdvTHU) = {

  proc main() : bool = {
    var pk, sk, y, t;

    (pk, sk) <$ keyGen;
    (y,  t)  =  A.forge(pk);

    return H (tagGen sk t) = y;
  }
}.


(* Phantom-Freeness *)
module type AdvPF(TagO : TagOracleT) = {
  proc forge(pk : pkey) : tag * int  {TagO.genTag}
}.

module GamePF(TagO : TagOracleT, A : AdvPF) = {
  module A = A(TagO)

  proc main() : bool = {
    var pk, sk, forged, tg, t;
   
    (pk, sk) <$ keyGen;    
    TagO.init(pk, sk);
    
    (tg, t) = A.forge(pk);
    forged = TagO.verTag(tg, t);    

    return forged /\ tg <> tagGen sk t;
  }
}.


end TagSystem.
